/*
 * @(#)CSSRule.java  1.0  6. Juni 2006
 *
 * Copyright (c) 2006 Werner Randelshofer (FHZ)
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */

package ch.randelshofer.swingcss;

import java.util.*;
import javax.swing.*;
/**
 * CSSRule matches on a JComponent class and on client properties.
 * <p>
 * Supported selectors:
 * <ul>
 * <li><code>*</code> matches all objects.</li>
 * <li><code>name</code> matches a fully qualified Java Class Name.</li>
 * <li><code>.name</code> matches the value of client property "css.class"</li>
 * <li><code>#name</code> matches the value of client property "css.id".</li>
 * </ul>
 *
 *
 * @author Werner Randelshofer
 * @version 1.0 6. Juni 2006 Created.
 */
public class CSSRule extends BasicRule {
    private String selector;
    private static enum SelectorType {
        ALL, CLASS, CLASS_PROPERTY, ID_PROPERTY
    }
    private SelectorType type;
    
    public CSSRule(String selector, String property, String value) {
        super(property, value);
        switch (selector.charAt(0)) {
            case '*' : type = SelectorType.ALL; break;
            case '.' : type = SelectorType.CLASS_PROPERTY; break;
            case '#' : type = SelectorType.ID_PROPERTY; break;
            default : type = SelectorType.CLASS; break;
        }
        this.selector = (type == SelectorType.CLASS) ? selector : selector.substring(1);
    }
    
    
    public boolean matches(Object obj) {
        boolean isMatch = false;
        switch (type) {
            case ALL :
                isMatch = true;
                break;
            case CLASS : {
                try {
                    isMatch = Class.forName(selector).isInstance(obj);
                } catch (ClassNotFoundException ex) {
                    // selector is not a fully qualified class name
                    // match class by name
                    String name = obj.getClass().getName();
                    int p = name.lastIndexOf('.');
                    if (p != -1) {
                        name = name.substring(p + 1);
                    }
                    isMatch = name.toLowerCase().equals(selector.toLowerCase());
                }
                break;
            }
            case CLASS_PROPERTY : {
                if (obj instanceof JComponent) {
                    JComponent c = (JComponent) obj;
                    String name = (String) c.getClientProperty("css.class");
                    isMatch = name != null && name.equals(selector);
                }
                break;
            }
            case ID_PROPERTY : {
                if (obj instanceof JComponent) {
                    JComponent c = (JComponent) obj;
                    String name = (String) c.getClientProperty("css.id");
                    isMatch = name != null && name.equals(selector);
                }
                break;
            }
        }
        return isMatch;
    }
    
    public String toString() {
        return "CSSRule["+selector+"{"+property+":"+value+"]";
    }
}